<?php

class Navigator
{
    private static $schemaReady = false;

    public static function dashboardStats()
    {
        self::ensureNavSchema();
        $db = Database::getInstance();
        $total = $db->fetch("SELECT COUNT(*) AS c FROM nav_items");
        $visible = $db->fetch("SELECT COUNT(*) AS c FROM nav_items WHERE is_visible = 1");
        $hidden = $db->fetch("SELECT COUNT(*) AS c FROM nav_items WHERE is_visible = 0");
        $latest = $db->fetchAll("SELECT id, title, url, created_at FROM nav_items ORDER BY id DESC LIMIT 5");

        return array(
            "total" => (int)$total["c"],
            "visible" => (int)$visible["c"],
            "hidden" => (int)$hidden["c"],
            "latest" => $latest
        );
    }

    public static function getSettings()
    {
        $db = Database::getInstance();
        $rows = $db->fetchAll("SELECT key_name, value FROM settings");
        $result = array();
        foreach ($rows as $row) {
            $result[$row["key_name"]] = $row["value"];
        }
        return $result;
    }

    public static function saveSettings($settings)
    {
        $allowed = array("site_name", "site_subtitle", "site_description", "theme_template", "copyright_text", "snow_enabled", "qq_number", "qq_group");
        $db = Database::getInstance();

        foreach ($allowed as $key) {
            if (!isset($settings[$key])) {
                continue;
            }
            $value = trim((string)$settings[$key]);
            if ($key === "theme_template" && !in_array($value, array("style1", "style2", "style3", "style15"), true)) {
                $value = "style2";
            }
            if ($key === "snow_enabled") {
                $value = $value === "1" ? "1" : "0";
            }

            $updated = $db->execute(
                "UPDATE settings SET value = :value WHERE key_name = :key_name",
                array(
                    "key_name" => $key,
                    "value" => $value
                )
            );
            if ($updated === 0) {
                $db->execute(
                    "INSERT INTO settings (key_name, value) VALUES (:key_name, :value)",
                    array(
                        "key_name" => $key,
                        "value" => $value
                    )
                );
            }
        }

        return true;
    }

    public static function listNav($keyword = "")
    {
        self::ensureNavSchema();
        $db = Database::getInstance();
        if ($keyword !== "") {
            $kw = "%" . $keyword . "%";
            return $db->fetchAll(
                "SELECT * FROM nav_items WHERE title LIKE :kw OR url LIKE :kw ORDER BY sort_order DESC, id DESC",
                array("kw" => $kw)
            );
        }
        return $db->fetchAll("SELECT * FROM nav_items ORDER BY sort_order DESC, id DESC");
    }

    public static function getNavById($id)
    {
        self::ensureNavSchema();
        $db = Database::getInstance();
        return $db->fetch("SELECT * FROM nav_items WHERE id = :id LIMIT 1", array("id" => (int)$id));
    }

    public static function addNav($data)
    {
        self::ensureNavSchema();

        $title = trim((string)$data["title"]);
        $url = trim((string)$data["url"]);
        $icon = trim((string)$data["icon"]);
        $iconType = isset($data["icon_type"]) ? trim((string)$data["icon_type"]) : "";
        $description = trim((string)$data["description"]);
        $sortOrder = isset($data["sort_order"]) ? (int)$data["sort_order"] : 0;
        $isVisible = !empty($data["is_visible"]) ? 1 : 0;

        if ($iconType === "") {
            $iconType = self::inferIconTypeByIcon($icon);
        }

        self::validateNav($title, $url, $icon, $iconType);

        $db = Database::getInstance();
        $now = date("Y-m-d H:i:s");
        $db->execute(
            "INSERT INTO nav_items (title, url, icon, icon_type, description, sort_order, is_visible, created_at, updated_at)
            VALUES (:title, :url, :icon, :icon_type, :description, :sort_order, :is_visible, :created_at, :updated_at)",
            array(
                "title" => $title,
                "url" => $url,
                "icon" => $icon,
                "icon_type" => $iconType,
                "description" => $description,
                "sort_order" => $sortOrder,
                "is_visible" => $isVisible,
                "created_at" => $now,
                "updated_at" => $now
            )
        );

        return $db->lastInsertId();
    }

    public static function updateNav($id, $data)
    {
        self::ensureNavSchema();

        $id = (int)$id;
        if ($id <= 0) {
            throw new Exception("参数错误");
        }

        $title = trim((string)$data["title"]);
        $url = trim((string)$data["url"]);
        $icon = trim((string)$data["icon"]);
        $iconType = isset($data["icon_type"]) ? trim((string)$data["icon_type"]) : "";
        $description = trim((string)$data["description"]);
        $sortOrder = isset($data["sort_order"]) ? (int)$data["sort_order"] : 0;
        $isVisible = !empty($data["is_visible"]) ? 1 : 0;

        if ($iconType === "") {
            $iconType = self::inferIconTypeByIcon($icon);
        }

        self::validateNav($title, $url, $icon, $iconType);

        $db = Database::getInstance();
        $db->execute(
            "UPDATE nav_items SET
                title = :title,
                url = :url,
                icon = :icon,
                icon_type = :icon_type,
                description = :description,
                sort_order = :sort_order,
                is_visible = :is_visible,
                updated_at = :updated_at
            WHERE id = :id",
            array(
                "title" => $title,
                "url" => $url,
                "icon" => $icon,
                "icon_type" => $iconType,
                "description" => $description,
                "sort_order" => $sortOrder,
                "is_visible" => $isVisible,
                "updated_at" => date("Y-m-d H:i:s"),
                "id" => $id
            )
        );

        return true;
    }

    public static function deleteNav($id)
    {
        self::ensureNavSchema();
        $db = Database::getInstance();
        return $db->execute("DELETE FROM nav_items WHERE id = :id", array("id" => (int)$id));
    }

    public static function batchDelete($ids)
    {
        self::ensureNavSchema();

        if (!is_array($ids) || empty($ids)) {
            throw new Exception("请选择要删除的数据");
        }

        $validIds = array();
        foreach ($ids as $id) {
            $id = (int)$id;
            if ($id > 0) {
                $validIds[] = $id;
            }
        }

        if (empty($validIds)) {
            throw new Exception("未找到有效数据");
        }

        $db = Database::getInstance();
        $placeholders = implode(",", array_fill(0, count($validIds), "?"));
        $db->execute("DELETE FROM nav_items WHERE id IN (" . $placeholders . ")", $validIds);
        return true;
    }

    public static function frontLinks()
    {
        self::ensureNavSchema();
        $db = Database::getInstance();
        return $db->fetchAll("SELECT * FROM nav_items WHERE is_visible = 1 ORDER BY sort_order DESC, id DESC");
    }

    private static function validateNav($title, $url, $icon, $iconType)
    {
        if ($title === "") {
            throw new Exception("网站名称不能为空");
        }
        if ($url === "") {
            throw new Exception("网站链接不能为空");
        }
        if (!preg_match("#^https?://#i", $url)) {
            throw new Exception("链接必须以 http:// 或 https:// 开头");
        }
        if (!in_array($iconType, array("font", "image"), true)) {
            throw new Exception("图标类型无效");
        }
        if ($iconType === "image" && $icon !== "" && !preg_match("#^(https?://|/|data:image/)#i", $icon)) {
            throw new Exception("图片图标请填写图片链接，或先上传图片");
        }
    }

    private static function inferIconTypeByIcon($icon)
    {
        return preg_match("#^(https?://|/|data:image/)#i", (string)$icon) ? "image" : "font";
    }

    private static function ensureNavSchema()
    {
        if (self::$schemaReady) {
            return;
        }

        $db = Database::getInstance();
        $columns = $db->fetchAll("PRAGMA table_info(nav_items)");
        if (empty($columns)) {
            self::$schemaReady = true;
            return;
        }

        $hasIconType = false;
        foreach ($columns as $column) {
            if (isset($column["name"]) && $column["name"] === "icon_type") {
                $hasIconType = true;
                break;
            }
        }

        if (!$hasIconType) {
            $db->execute("ALTER TABLE nav_items ADD COLUMN icon_type TEXT NOT NULL DEFAULT 'font'");
        }

        $db->execute(
            "UPDATE nav_items SET icon_type = CASE
                WHEN icon_type = 'image' THEN 'image'
                WHEN icon_type = 'font' THEN 'font'
                WHEN icon LIKE 'http://%' OR icon LIKE 'https://%' OR icon LIKE '/%' OR icon LIKE 'data:image/%' THEN 'image'
                ELSE 'font'
            END"
        );

        self::$schemaReady = true;
    }
}
